<?php

namespace App\Http\Controllers\Booking;

use App\Http\Controllers\Controller;
use App\Models\BuyProduct;
use App\Models\BuyProductDetail;
use App\Models\Product;
use App\Models\Worker;
use Illuminate\Http\Request;
use DataTables;
use DB;
use Dompdf\Dompdf;
use Dompdf\Options;
use PDF;
use Illuminate\Support\Facades\Auth;

class BuyProductController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = BuyProduct::select('*')->with("sales_employee_user");
            if (get_user_role() == 2) {
                $data->whereHas('created_by_user', function ($query) {
                    return $query->where('branch_id', Auth::user()->branch_id);
                });
            }
            return Datatables::of($data)
                ->addIndexColumn()
                ->addColumn('date', function ($row) {
                    return date('d/m/Y H:i:s', strtotime($row->created_at));
                })
                ->addColumn('created_by', function ($row) {
                    return (!empty($row->created_by_user)) ? $row->created_by_user->first_name . ' ' . $row->created_by_user->last_name : '';
                })
                ->addColumn('sales_employee', function ($row) {
                    if(isset($row->sales_employee_user))
                    {
                        return (!empty($row->sales_worker_id)) ? $row->sales_employee_user->name : '';
                    }
                    else return 'Not Exists';
                })
                ->addColumn('products', function ($row) {
                    $template = "";
                    if (!$row->details->isEmpty()) {
                        foreach ($row->details as $detail) {
                            if (!empty($detail->product)) {
                                $template .= $detail->product->name_en . "<br>";
                            }
                        }
                    }
                    return $template;
                })
                ->addColumn('total', function ($row) {
                    $total = 0;
                    if (!$row->details->isEmpty()) {
                        foreach ($row->details as $detail) {
                            $total += $detail->price;
                        }
                    }
                    return get_currency() . get_num_format($total);
                })
                ->addColumn('action', function ($row) {
                    $btn = "<a data-id='" . $row->id . "' href='" . set_route('buy.product.print') . "?id=" . $row->id . "' target='_blank'  class='btn btn-primary btn-sm print-row'><i class='fa fa-print'></i></a>";
                    if (can('discount.delete')) {
                        $btn .= '<a href="' . set_route('buy.product.delete', ['id' => $row->id]) . '" class="btn btn-danger btn-xs delete-row">
                                    <i class="fa fa-close"></i>
                                </a>';
                    }
                    return $btn;
                })
                ->rawColumns(['action', 'products'])
                ->make(true);
        }
        $products = Product::all();
        if (Auth()->user()->branch_id)
            $workers = Worker::where("branch_id", Auth()->user()->branch_id)->get();
        else
            $workers = Worker::all();

        return view('buy_product.index', compact('products', 'workers'));
    }
    public function store(Request $request)
    {
        return DB::transaction(function () use ($request) {
            if (isset($request->product_id) && !empty($request->product_id)) {
                $commission = $request->get('commission');
                $discount = $request->get('discount');
                if (empty($request->worker_id) || !in_array($commission, commission_list())) {
                    $commission = 0;
                }
                if (!in_array($discount, discount_list())) {
                    $discount = 0;
                }
                $buy_product = BuyProduct::create([
                    'created_by' => Auth::id(),
                    'payment_type' => $request->get('payment_type'),
                    'worker_id' => $request->get('worker_id'),
                    'sales_worker_id' => $request->get('sales_worker_id'),
                    'commission' => $commission,
                    'discount' => $discount
                ]);
                foreach ($request->product_id as $item) {
                    $price = 0;
                    $product = Product::find($item);
                    if (!empty($product)) {
                        $price = $product->price;
                    }
                    BuyProductDetail::create([
                        'buy_product_id' => $buy_product->id,
                        'product_id' => $item,
                        'price' => $price,
                    ]);
                }
            }
            set_message('Order was saved.');
            return response()->json(['success' => true, 'message' => 'Order was saved.']);
        });
    }
    //delete
    public function destroy(Request $request)
    {
        can('discount.delete', true);
        BuyProductDetail::where('buy_product_id', $request->get('id'))->delete();
        $row = BuyProduct::findOrFail($request->get('id'));
        $row->deleted_by = Auth::id();
        $row->save();
        $row->delete();
        set_message('Order was delete successfully.');
        return redirect()->back();
    }
    public function print_invoice(Request $request)
    {
        $result = BuyProduct::with("created_by_user")->findOrFail($request->id);

        $pdf = new Dompdf();
        $pdf->setPaper(array(0, 0, 226, 800));
        $GLOBALS['bodyHeight'] = 0;
        $pdf->setCallbacks(
            array(
                'myCallbacks' => array(
                    'event' => 'end_frame', 'f' => function ($infos) {
                        $frame = $infos["frame"];
                        if (strtolower($frame->get_node()->nodeName) === "body") {
                            $padding_box = $frame->get_padding_box();
                            $GLOBALS['bodyHeight'] += $padding_box['h'];
                        }
                    }
                )
            )
        );
        $pdf->loadHtml(view('buy_product.invoice', compact('result'))->render());
        $pdf->render();
        unset($pdf);
        $pdf = PDF::loadView('buy_product.invoice', compact('result'));
        // $pdf->dompdf->set_option('dpi' , '512');
        $pdf->setPaper(array(0, 0, 226, $GLOBALS['bodyHeight'] + 50));
        return $pdf->stream('pdfview.pdf');
    }
}
