<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use App\User;

class ResetPasswordController extends Controller
{
    // عرض نموذج إعادة تعيين كلمة المرور
    public function showResetForm($token)
    {
        Log::info('Password reset form requested', [
            'token' => $token,
            'ip' => request()->ip(),
            'user_agent' => request()->userAgent()
        ]);
        
        try {
            // Decode the token to get email
            $decoded = base64_decode($token);
            $parts = explode('|', $decoded);
            
            if (count($parts) !== 2) {
                Log::warning('Invalid token format', ['token' => $token]);
                return redirect('login')->with('error', 'Invalid reset link.');
            }
            
            $email = $parts[0];
            $timestamp = $parts[1];
            
            // Check if link is expired (60 minutes)
            if (time() - $timestamp > 3600) {
                Log::warning('Reset link expired', ['email' => $email, 'timestamp' => $timestamp]);
                return redirect('login')->with('error', 'Reset link has expired. Please request a new one.');
            }
            
            // Check if email exists
            $user = User::where('email', $email)->first();
            if (!$user) {
                Log::warning('User not found for reset', ['email' => $email]);
                return redirect('login')->with('error', 'User not found.');
            }
            
            Log::info('Reset form displayed successfully', [
                'email' => $email,
                'user_id' => $user->id
            ]);
            
            return view('auth.passwords.reset', [
                'token' => $token, 
                'email' => $email
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error processing reset token', [
                'token' => $token,
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            return redirect('login')->with('error', 'Invalid reset link.');
        }
    }

    // إعادة تعيين كلمة المرور
    public function reset(Request $request)
    {
        Log::info('Password reset attempt started', [
            'email' => $request->email,
            'token' => $request->token,
            'ip' => request()->ip()
        ]);
        
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|confirmed|min:6',
        ]);

        try {
            // Decode and validate token
            $decoded = base64_decode($request->token);
            $parts = explode('|', $decoded);
            
            if (count($parts) !== 2) {
                Log::warning('Invalid token format in reset', ['token' => $request->token]);
                return back()->withInput()->with('error', 'Invalid reset link.');
            }
            
            $tokenEmail = $parts[0];
            $timestamp = $parts[1];
            
            // Check if link is expired (60 minutes)
            if (time() - $timestamp > 3600) {
                Log::warning('Reset link expired in reset', ['email' => $tokenEmail, 'timestamp' => $timestamp]);
                return back()->withInput()->with('error', 'Reset link has expired. Please request a new one.');
            }
            
            // Check if email matches token
            if ($tokenEmail !== $request->email) {
                Log::warning('Email mismatch in reset', [
                    'token_email' => $tokenEmail,
                    'request_email' => $request->email
                ]);
                return back()->withInput()->with('error', 'Email mismatch.');
            }

            // التحقق مما إذا كان البريد الإلكتروني موجود في قاعدة البيانات
            $user = User::where('email', $request->email)->first();

            if (!$user) {
                Log::warning('Password reset failed - User not found', [
                    'email' => $request->email,
                    'ip' => request()->ip()
                ]);
                return back()->withInput()->with('error', 'Invalid email.');
            }

            Log::info('User found for password reset', [
                'user_id' => $user->id,
                'email' => $user->email
            ]);

            // تحديث كلمة المرور
            $user->password = Hash::make($request->password);
            $user->save();

            Log::info('Password reset successful', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip' => request()->ip()
            ]);

            // تسجيل الدخول تلقائيًا بعد إعادة تعيين كلمة المرور
            auth()->login($user);

            return redirect('/login')->with('success', 'Password has been reset successfully!');
            
        } catch (\Exception $e) {
            Log::error('Error during password reset', [
                'email' => $request->email,
                'token' => $request->token,
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            return back()->withInput()->with('error', 'An error occurred. Please try again.');
        }
    }
}
