<?php

namespace App\Http\Controllers\Worker;

use App\Models\Branch;
use App\Models\Service;
use App\Models\Shift;
use App\Models\Vacation;
use App\Models\Worker;
use App\Models\Role;
use App\Models\WorkerService;
use App\User;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use DataTables;

class WorkerController extends Controller
{
    /**
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {

        can('worker.view', true);
        if ($request->ajax()) {
            $data = Worker::select('*')->with('services');
            if (auth()->user()->role_id == 2)
                $data = Worker::select('*')->with('services')->where("branch_id", auth()->user()->branch_id);
            return Datatables::of($data)
                ->addIndexColumn()
                ->addColumn('phone', function ($row) {
                    return "+971" . $row->phone;
                })
                ->addColumn('image', function ($row) {
                    $btn = "<img width='50' src='" . url('assets/img/avatar3_small.jpg') . "'>";
                    if (!empty($row->image)) {
                        $btn = '<img width="50" src="' . url(\Storage::url($row->image)) . '" />';
                    }
                    return $btn;
                })
                ->addColumn('services', function ($row) {
                    $template = "";
                    if (!$row->services->isEmpty()) {
                        foreach ($row->services as $service) {
                            if (isset($service->service) && !empty($service->service)) {
                                $template .= $service->service->name_en . "<br>";
                            }
                        }
                    }
                    return $template;
                })
                ->addColumn('branch_id', function ($row) {
                    $template = "";
                    if (isset($row->branch) && !empty($row->branch)) {
                        $template = $row->branch->name_en;
                    }
                    return $template;
                })
                ->addColumn('action', function ($row) {
                    $btn = "";
                    if (can('worker.edit')) {
                        $btn .= '<a href="" class="btn btn-success btn-xs add-vacation" data-name="' . $row->name . '" data-id="' . $row->id . '" data-target="#vacationModal" data-toggle="modal">
                                <i class="fa fa-dashcube"></i>
                            </a>';
                    }
                    if (auth()->user()->role_id == 1) {
                        if (can('worker.edit')) {
                            $btn .= '<a href="" class="btn btn-primary btn-xs edit-worker" data-id="' . $row->id . '" data-target="#editModal" data-toggle="modal">
                                    <i class="fa fa-edit"></i>
                                </a>';
                        }
                        if (can('worker.delete')) {
                            $btn .= '<a href="' . set_route('worker.delete', ['id' => $row->id]) . '" class="btn btn-danger btn-xs delete-row">
                                        <i class="fa fa-close"></i>
                                    </a>';
                        }
                    }
                    return $btn;
                })
                ->rawColumns(['action', 'image', 'services'])
                ->make(true);
        }
        $branches = Branch::where('status', 'Active')->get();
        $services = Service::where('status', 'Active')->get();
        $shifts = Shift::where('status', 'Active')->get();
        return view('worker.index', compact('branches', 'services', 'shifts'));
    }
    //create
    public function create(Request $request)
    {
        can('worker.create', true);
        $roles = Role::all();
        return view('worker.create', compact('roles'));
    }
    public function store(Request $request)
    {
        can('worker.create', true);
        $message_result = [];
        $validator = Validator::make($request->all(), [
            'name' => 'required'
        ]);
        if ($validator->fails()) {
            $messages = $validator->messages();
            foreach ($messages->all('<p>:message</p>') as $index => $message) {
                array_push($message_result, $message);
            }
            return response()->json(['success' => false, 'message' => $message_result]);
        } else {
            //check if phone has zero
            $first_phone = substr($request->get('phone'), 0, 1);
            if ($first_phone == "0" || $first_phone == 0) {
                return response()->json(['success' => false, 'message' => __('s.phone_zero')]);
            }

            $data = $request->all();
            $image = "";
            //upload image
            if ($request->file('image')) {
                $imagePath = $request->file('image');
                $imageName = time() . str_replace(' ', '', $imagePath->getClientOriginalName());
                $image = $request->file('file')->storeAs('workers/uploads', $imageName, 'public');
            }
            $data['image'] = $image;

            /*if(isset($data['has_commission'])){
                $data['has_commission'] = 1;
            }else{
                $data['has_commission'] = 0;
            }*/

            $worker = Worker::create($data);
            $this->save_services($request, $worker->id);
            set_message('Worker was saved successfully.');
        }
        return response()->json(['success' => true, 'message' => 'Worker was saved successfully.']);
    }
    //update
    public function edit(Request $request)
    {
        $row = Worker::find($request->get('id'));
        $services = [];
        if (!empty($row->services)) {
            foreach ($row->services as $service) {
                array_push($services, $service->service_id);
            }
        }
        return response()->json(['success' => true, 'row' => $row, 'service_id' => $services]);
    }
    public function update(Request $request)
    {
        can('worker.edit', true);
        $data = $request->all();
        if (isset($data['_token'])) {
            unset($data['_token']);
        }
        $worker = Worker::find($request->get('id'));
        //check if phone has zero
        $first_phone = substr($request->get('phone'), 0, 1);
        if ($first_phone == "0" || $first_phone == 0) {
            return response()->json(['success' => false, 'message' => __('s.phone_zero')]);
        }

        $image = "";
        //upload image
        if ($request->file('image')) {
            $imagePath = $request->file('image');
            $imageName = time() . str_replace(' ', '', $imagePath->getClientOriginalName());
            $image = $request->file('image')->storeAs('workers/uploads', $imageName, 'public');
            //delete old file
            Storage::disk('public')->delete($worker->image);
        } else {
            $image = $worker->image;
        }
        $data['image'] = $image;
        if (isset($data['service_id'])) {
            unset($data['service_id']);
        }
        /*if(isset($data['has_commission'])){
            $data['has_commission'] = 1;
        }else{
            $data['has_commission'] = 0;
        }*/

        Worker::where('id', $request->get('id'))->update($data);
        $this->save_services($request, $request->get('id'));
        return response()->json(['success' => true, 'message' => 'Worker was saved successfully.']);
    }
    //delete
    public function destroy(Request $request)
    {
        can('worker.delete', true);
        WorkerService::where('worker_id', $request->route('id'))->delete();
        $row = Worker::findOrFail($request->route('id'));
        $row->delete();
        set_message('Worker was delete successfully.');
        return redirect()->back();
    }

    public function save_services(Request $request, $worker_id)
    {
        WorkerService::where('worker_id', $worker_id)->delete();
        if (isset($request->service_id) && !empty($request->service_id)) {
            foreach ($request->service_id as $item) {
                WorkerService::create([
                    'service_id' => $item,
                    'worker_id' => $worker_id
                ]);
            }
        }
    }
    public function get_vacation(Request $request)
    {
        $worker_id = $request->get('id');
        $worker = Worker::find($worker_id);
        $view = (string)view('worker.vacation', compact('worker'));
        return response()->json(['success' => true, 'view' => $view]);
    }
    public function save_vacation(Request $request)
    {
        $data = $request->all();
        $data['day'] = convert_date($request->get('day'));
        Vacation::create($data);
        set_message('Vacation was saved successfully.');
        return response()->json(['success' => true]);
    }
}
