<?php

namespace App\Http\Controllers\User;

use App\Models\Role;
use App\User;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use DataTables;

class UserController extends Controller
{
    /**
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {

        can('user.view',true);
        if ($request->ajax()) {
            $data = User::where('id','!=',1)->select('*');
            return Datatables::of($data)
                ->addIndexColumn()
                ->addColumn('image',function ($row){
                    $btn = "<img width='50' src='assets/img/avatar3_small.jpg'>";
                    if(!empty($row->image)){
                        $btn = '<img width="50" src="'.url(\Storage::url($row->image)).'" />';
                    }
                    return $btn;
                })
                ->addColumn('action', function($row){
                    $btn = "";
                    if(can('user.edit')){
                        $btn.='<a href="'.url('user/'.$row->id.'/edit').'" class="btn btn-primary btn-xs">
                                    <i class="fa fa-edit"></i>
                                </a>';
                    }
                    if(can('user.delete')){
                        $btn.='<a href="'.url('user/'.$row->id.'/delete').'" class="btn btn-danger btn-xs delete-row">
                                    <i class="fa fa-close"></i>
                                </a>';
                    }
                    return $btn;
                })
                ->rawColumns(['action','image'])
                ->make(true);
        }
        return view('user.index');
    }
    //create
    public function create(Request $request)
    {
        can('user.create',true);
        $roles = Role::all();
        return view('user.create',compact('roles'));
    }
    public function store(Request $request){
        can('user.create',true);
        $message_result = [];
        $validator = Validator::make($request->all(), [
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'email:rfc|unique:users',
            'role_id' => 'required',
            'password' => 'required|confirmed',
            'file' => 'mimes:jpg,jpeg,png|max:1024'
        ]);
        if ($validator->fails()) {
            $messages = $validator->messages();
            foreach ($messages->all('<p>:message</p>') as $index => $message) {
                array_push($message_result, $message);
            }
            return response()->json(['success' => false, 'message' => $message_result]);
        }else{
            $image = "";
            //upload image
            if($request->file('file')){
                $imagePath = $request->file('file');
                $imageName = time().str_replace(' ', '', $imagePath->getClientOriginalName());
                $image = $request->file('file')->storeAs('users/uploads', $imageName, 'public');
            }
            User::create([
                'first_name'=>$request->get('first_name'),
                'last_name'=>$request->get('last_name'),
                'email'=>$request->get('email'),
                'phone'=>$request->get('phone'),
                'password'=>Hash::make($request->get('password')),
                'image'=>$image,
                'role_id'=>$request->get('role_id')
            ]);
            set_message('User was saved successfully.');
        }
        return response()->json(['success'=>true,'message'=>'User was created successfully.']);
    }
    //update
    public function edit($id = 0,Request $request){
        can('user.edit',true);
        $roles = Role::all();
        $user = User::findOrFail($id);
        return view('user.create',compact('user','roles'));
    }
    public function update($id,Request $request){
        can('user.edit',true);
        $user = User::find($id);
        $message_result = [];
        $validation_array = [
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'required|email:rfc',
            'role_id' => 'required',
            'file' => 'mimes:jpg,jpeg,png|max:1024'
        ];
        //check for password
        if(!empty($request->get('password'))){
            $validation_array['password'] = 'required|confirmed';
        }
        //check for email
        $is_email_exsit = false;
        $get_user = User::where('email','=',trim($request->get('email')))
            ->where('id','!=',$id)
            ->get()->first();
        if(!empty($get_user)){
            $is_email_exsit = true;
        }

        //check for validation
        $validator = Validator::make($request->all(), $validation_array);
        if ($validator->fails()) {
            $messages = $validator->messages();
            foreach ($messages->all('<p>:message</p>') as $index => $message) {
                array_push($message_result, $message);
            }
            return response()->json(['success' => false, 'message' => $message_result]);
        }
        elseif ($is_email_exsit){
            return response()->json(['success'=>false,'message'=>'<p>The email has already been taken.</p>']);
        }
        else{
            $image = "";
            //upload image
            if($request->file('file')){
                $imagePath = $request->file('file');
                $imageName = time().str_replace(' ', '', $imagePath->getClientOriginalName());
                $image = $request->file('file')->storeAs('users/uploads', $imageName,'public');
                //delete old file
                Storage::disk('public')->delete($user->image);
            }else{
                $image = $user->image;
            }
            //get all request
            $all_request = $request->all();
            if(isset($all_request['_token'])){
                unset($all_request['_token']);
            }
            //check password
            if(!empty($request->get('password'))){
                $all_request['password'] = Hash::make($request->get('password'));
            }else{
                unset($all_request['password']);
                unset($all_request['password_confirmation']);
            }
            $all_request['image'] = $image;
            if(isset($all_request['file'])){
                unset($all_request['file']);
            }
            $user->update($all_request);
            set_message('User was saved successfully.');
        }
        return response()->json(['success'=>true,'message'=>'User was saved successfully.']);
    }
    //delete
    public function destroy($id,Request $request){
        can('user.delete',true);
        $user = User::findOrFail($id);
        $user->delete();
        set_message('User was delete successfully.');
        return redirect()->back();
    }
}
